/*
 * Convert among DOS, Mac, and Unix formats.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

/* Get setmode() and O_BINARY if they are available. */
#if defined (_WIN32) || defined (__CYGWIN__)
#   include <io.h>
#   include <fcntl.h>
#endif

/* Because stdout will be in binary mode, you have to make sure the
 * correct EOL characters are written yourself.  The exception is the
 * help output because it is printed before converting stdout and
 * stdin to O_BINARY mode. */
#if defined (_WIN32) || defined (__CYGWIN__)
static char* eol = "\r\n";
#else
static char* eol = "\n";
#endif

static int verbose = 0;
static const char* _progname = NULL;
static sigset_t handled_signals;
static const char* current_tmp_filename = NULL;
static void (*ConvertAndPutChar)(int ch, FILE* f) = NULL;

static const char* GetProgname()
{
  return _progname;
}

static void SetProgname(const char* progname)
{
  _progname = progname;
}

static void Usage(int exit_value)
{
  FILE* f = stdout;
  if( exit_value ) {
    f = stderr;
  }
  fprintf(f, "\n");
  fprintf(f, "Usage: %s [options] [<file> . . .]\n", GetProgname());
  fprintf(f, "\n");
  fprintf(f, "       General Options\n");
  fprintf(f, "        -h or --help    : Show this help.\n");
  fprintf(f, "        -v or --verbose : Report files on stdout.\n");
  fprintf(f, "\n");
  fprintf(f, "       Conversion Options\n");
  fprintf(f, "        -d or --crlf    : Convert to DOS format.\n");
  fprintf(f, "        -m or --cr      : Convert to Mac format.\n");
  fprintf(f, "        -u or --lf      : Convert to Unix format.\n");
  fprintf(f, "\n");
  fprintf(f, "       Instructions\n");
  fprintf(f, "        The default conversion depends on the name used to\n");
  fprintf(f, "        start the program.  If you invoke it as \"crlf\", DOS\n");
  fprintf(f, "        conversion is the default.  If you invoke it as "
          "\"cr\",\n");
  fprintf(f, "        Mac conversion is the default.  If you invoke it as\n");
  fprintf(f, "        \"lf\", Unix conversion is the default.\n");
  fprintf(f, "\n");
  fprintf(f, "        If you don't specify a file or specify the \"-\" file,\n");
  fprintf(f, "        standard input and standard output is used.  For all\n");
  fprintf(f, "        other files, the conversion is safely performed\n");
  fprintf(f, "        in-place.\n");
  fprintf(f, "\n");
  fprintf(f, "Version: 2.3\n");
  fprintf(f, "\n");

  exit(exit_value);
}

/* Basename() does not allocate a new string.  Rather, it simply
 * returns a pointer that points to the basename embedded in
 * "fullpath". */
static const char* Basename(const char* fullpath)
{
  const char* rv = strrchr(fullpath, (int)'/');
  if( rv == NULL ) {
    rv = fullpath;
  } else {
    ++rv;
  }
  return rv;
}

static void (*Signal(int sig, void (*handler)(int)))(int)
{
  void (*rv)(int);
  if( (rv = signal(sig, handler)) == SIG_ERR ) {
    fprintf(stderr, "%s: signal: %s%s", GetProgname(), strerror(errno), eol);
    exit(1);
  }
  return rv;
}

static void FPutC(int ch, FILE* f)
{
  while( fputc(ch, f) == EOF ) {
    fprintf(stderr, "%s: fputc: %s%s", GetProgname(), strerror(errno), eol);
    exit(1);
  }
}

static int FGetC(FILE* f)
{
  int ch;
  errno = 0;
  while( (ch = fgetc(f)) == EOF ) {
    if( errno == 0 ) {
      /* EOF */
      break;
    }
    fprintf(stderr, "%s: fgetc: %s%s", GetProgname(), strerror(errno), eol);
    exit(1);
  }
  return ch;
}

static void ForceBinaryMode(FILE* f)
{
#ifdef O_BINARY
    /* 
     * Take into account the fact that Windows is designed to only
     * pipe text streams, i.e., stdin is opened by default in "text"
     * mode.  We need to put it in "binary" mode without losing data.
     * The old-style C++ IO library had ifstream::setmode()
     * ofstream::setmode() calls.  I don't know where they went.  So,
     * I'm left with altering the underlying file descriptor for what
     * will usually be a buffered stream because this program will
     * usually be used as part of a shell pipe or redirection.
     */
    int fd = fileno(f);
    if( fd < 0 ) {
      fprintf(stderr,
              "ForceBinaryMode(): Invalid file descriptor: %d%s",
              fd,
              eol);
      exit(1);
    }
    /* Warning: BSD defines setmode() too, but it does something
     * different. */
    if( setmode(fd, O_BINARY) == -1 ) {
      fprintf(stderr, "setmode(%d): %s%s", fd, strerror(errno), eol);
      exit(1);
    }
#endif    
}

static void ConvertAndPutDosChar(int ch, FILE* f)
{
  if( (ch == '\r') || (ch == '\n') ) {
    FPutC('\r', f);
    FPutC('\n', f);
  } else {
    FPutC(ch, f);
  }
}

static void ConvertAndPutMacChar(int ch, FILE* f)
{
  if( ch == '\n' ) {
    FPutC('\r', f);
  } else {
    FPutC(ch, f);
  }
}

static void ConvertAndPutUnixChar(int ch, FILE* f)
{
  if( ch == '\r' ) {
    FPutC('\n', f);
  } else {
    FPutC(ch, f);
  }
}

static void SigEmptySet(sigset_t* sset)
{
  if( sigemptyset(sset) == -1 ) {
    fprintf(stderr,
            "%s: sigemptyset: %s%s",
            GetProgname(),
            strerror(errno),
            eol);
    exit(1);
  }
}

static void SigAddSet(sigset_t* set, int sig)
{
  if( sigaddset(set, sig) == -1 ) {
    fprintf(stderr,
            "%s: sigaddset: %s%s",
            GetProgname(),
            strerror(errno),
            eol);
    exit(1);
  }
}

static void SigProcMask(int mode, const sigset_t* sset, sigset_t* osset)
{
  if( sigprocmask(mode, sset, osset) == -1 ) {
    fprintf(stderr,
            "%s: sigprocmask: %s%s",
            GetProgname(),
            strerror(errno),
            eol);
    exit(1);
  }
}

static void SignalHandler(int sig)
{
  /* Error checking is light given that there isn't much I can do from
   * within the signal handler and that I'm about to exit anyway. */
  if( current_tmp_filename ) {
    unlink(current_tmp_filename);
  }
  _exit(1);
}

static void SignalInit()
{
  SigEmptySet(&handled_signals);
  SigAddSet(&handled_signals, SIGHUP);
  SigAddSet(&handled_signals, SIGINT);
  SigAddSet(&handled_signals, SIGQUIT);
  SigAddSet(&handled_signals, SIGTERM);
  Signal(SIGHUP, SignalHandler);
  Signal(SIGINT, SignalHandler);
  Signal(SIGQUIT, SignalHandler);
  Signal(SIGTERM, SignalHandler);
}

static FILE* OpenFileToConvert(const char* filename)
{
  FILE* rv = NULL;
  if( strcmp(filename, "-") == 0 ) {
    rv = stdin;
  } else {
    rv = fopen(filename, "rb+");
  }
  return rv;
}

/* OpenTmpFile() creates a temporary file based on "filename" so that
 * the temporary file will reside in the same directory as "filename"
 * so that rename() will just be a rename() instead of a copy.  The
 * temporary file name is returned in tmpfilename if the temporary
 * file was created.  The return value is NULL if the temporary wasn't
 * created.  Otherwise, the return value has a "FILE*" to the newly
 * created temporary file. */
static FILE* OpenTmpFile(const char* filename, char** tmpfilename)
{
  FILE* f = NULL;
  int fd = -1;

  if( strcmp(filename, "-") == 0 ) {
    *tmpfilename = NULL;
    return stdout;
  }

  *tmpfilename = malloc(strlen(filename) + 7 + 1);
  if( tmpfilename == NULL ) {
    return NULL;
  }

  strcpy(*tmpfilename, filename);
  strcat(*tmpfilename, "_XXXXXX");

  fd = mkstemp(*tmpfilename);
  if( fd == -1 ) {
    free(*tmpfilename);
    *tmpfilename = NULL;
    return NULL;
  }

  f = fdopen(fd, "rb+");
  if( f == NULL ) {
    *tmpfilename = NULL;
    return NULL;
  }

  return f;
}

/* StreamFileWithConversion() streams the data in "f" to "ftmp"
 * converting on the fly. */
static void StreamFileWithConversions(FILE* f, FILE* ftmp)
{
  int chSeen = '\0';
  int chNew = '\0';

  /*
   * Prime the pump.  You need to be careful because the file could
   * contain just zero or one characters.
   */
 
  if( (chSeen = FGetC(f)) == EOF ) {
    /* File was empty. */
    return;
  }

  if( (chNew = FGetC(f)) == EOF ) {
    /* File contained one character. */
    ConvertAndPutChar(chSeen, ftmp);
    return;
  }

  /*
   * Main loop.
   */

  do {
    /* Telescope "\r\n" sequences to a single '\n' character.  You can
     * then treat all files as only having '\r' or '\n' as the EOL
     * character. */
    if( (chSeen == '\r') && (chNew == '\n') ) {
      chSeen = '\n';
      continue;
    }
    ConvertAndPutChar(chSeen, ftmp);
    chSeen = chNew;
  } while( (chNew = FGetC(f)) != EOF );

  /* Still need to write the last character. */
  ConvertAndPutChar(chSeen, ftmp);
}

static int RenameTemporaryFile(const char* filename, const char* ftmpname)
{
  int rv = 1;
  struct stat stat_rv;

  if( strcmp(filename, "-") != 0 ) {

    /* mkstemp() sets the original mode to 0600.  This needs to be
     * altered to be the same as the original file. */
    if( stat(filename, &stat_rv) == -1 ) {
      fprintf(stderr,
              "%s: Strange: Failed to stat the original file when "
              "renaming temporary.  stat: %s%s",
              GetProgname(),
              strerror(errno),
              eol);
      rv = 0;
    }

    if( rv && (rename(ftmpname, filename) == -1) ) {
      fprintf(stderr,
              "%s: Skipping \"%s\".  Conversion was o.k., "
              "but unable to rename temporary file.  rename: %s%s",
              GetProgname(),
              filename,
              strerror(errno),
              eol);
      if( unlink(ftmpname) == -1 ) {
        fprintf(stderr,
                "%s: Strange:  I'm unable to delete the temporary "
                "file \"%s\" that I just created!.  unlink: %s%s",
                GetProgname(),
                ftmpname,
                strerror(errno),
                eol);
      }
      rv = 0;
    }

    if( rv && (chmod(filename, stat_rv.st_mode) == -1) ) {
      fprintf(stderr,
              "%s: Warning: File permissions for %s will need manual "
              "adjustment.  chmod: %s%s",
              GetProgname(),
              filename,
              strerror(errno),
              eol);
    }

  }
  return rv;
}

static int ConvertFile(const char* filename)
{
  FILE* f;
  FILE* ftmp;
  char* ftmpname;
  int rv = 0;


  /* Open the original file. */
  f = OpenFileToConvert(filename);
  if( f == NULL ) {
    fprintf(stderr,
            "%s: Skipping \"%s\".  fopen: %s%s",
            GetProgname(),
            filename,
            strerror(errno),
            eol);
    return 0;
  }


  /* Open the temporary file, but this has to be done in an
   * asyncsignal-safe manner such that opening the temporary file and
   * setting current_tmp_filename are atomic from the signal handler's
   * perspective. */
  SigProcMask(SIG_BLOCK, &handled_signals, NULL);
  ftmp = OpenTmpFile(filename, &ftmpname);
  if( ftmp == NULL ) {
    fprintf(stderr,
            "%s: Skipping \"%s\".  Unable to create temporary file.%s",
            GetProgname(),
            filename,
            eol);
    fclose(f);
    SigProcMask(SIG_UNBLOCK, &handled_signals, NULL);
    return 0;
  }
  current_tmp_filename = ftmpname;
  SigProcMask(SIG_UNBLOCK, &handled_signals, NULL);


  /* Do the conversion. */
  StreamFileWithConversions(f, ftmp);


  /* Don't leave data in the C library's buffers.  This data would
   * otherwise be lost if a signal were to come at just the right
   * time. */
  fclose(ftmp);
  fclose(f);


  /* Remove original file and put the temporary file in its place.
   * You have to be careful to atomically update
   * current_tmp_filename. */
  SigProcMask(SIG_BLOCK, &handled_signals, NULL);
  rv = RenameTemporaryFile(filename, ftmpname);
  current_tmp_filename = NULL;
  SigProcMask(SIG_UNBLOCK, &handled_signals, NULL);

  /* We are responsible for ftmpname. */
  if( ftmpname ) {
    free(ftmpname);
    ftmpname = NULL;
  }

  return rv;
}


/*
 * Convert any instance of '\n' or '\r' to "\r\n".
 */
int main(int argc, char* argv[])
{
  int i = 1;

  SetProgname(Basename(argv[0]));

  /*
   * Signal handlers.
   */
  SignalInit();
  
  /* Multi-call binary: Default to "\r\n" (dos conversion) in honor of
   * the program's name.  If called as "lf", unix conversion is
   * desired.  If called as "cr", mac converstion is desired. */
  ConvertAndPutChar = ConvertAndPutDosChar;
  if( strcmp(GetProgname(), "lf") == 0 ) {
    ConvertAndPutChar = ConvertAndPutUnixChar;
  } else if( strcmp(GetProgname(), "cr") == 0 ) {
    ConvertAndPutChar = ConvertAndPutMacChar;
  }

  /* Handle command-line parameters. */
  for( i = 1 ; i < argc ; ++i ) {
    char* arg = argv[i];
    if( arg[0] != '-' ) {
      break;
    }
    if( (strcmp(arg, "-d") == 0) || (strcmp(arg, "--crlf") == 0) ) {
      ConvertAndPutChar = ConvertAndPutDosChar;
    } else if( (strcmp(arg, "-h") == 0) || (strcmp(arg, "--help") == 0) ) {
      Usage(0);
    } else if( (strcmp(arg, "-m") == 0) || (strcmp(arg, "--cr") == 0) ) {
      ConvertAndPutChar = ConvertAndPutMacChar;
    } else if( (strcmp(arg, "-u") == 0) || (strcmp(arg, "--lf") == 0) ) {
      ConvertAndPutChar = ConvertAndPutUnixChar;
    } else if( (strcmp(arg, "-v") == 0) || (strcmp(arg, "--verbose") == 0) ) {
      verbose = 1;
    } else {
      Usage(1);
    }
  }

  /*
   * Input could be read from stdin and written to stdout depending on
   * how the user invokes the program.  So, put them in binary mode.
   * You want to wait until after Usage() could be called because you
   * want stdout to still be in text mode at that point.
   */
  ForceBinaryMode(stdin);
  ForceBinaryMode(stdout);

  if( i == argc ) {
    /* No files were specified on the command line.  So, operate on
     * stdin and stdout. */
    ConvertFile("-");
  }

  /* Handle the files specified on the command line. */  
  for( ; i < argc ; ++i ) {
    int cf_rv = 0;
    if( verbose ) {
      printf("Converting %s . . . ", argv[i]); fflush(stdout);
    }
    cf_rv = ConvertFile(argv[i]);
    if( verbose ) {
      printf("%s%s", cf_rv ? "Success." : "Error.", eol);
      fflush(stdout);
    }
  }

  return 0;
}
