#!/usr/bin/env python

"""
Test pathmgr by shelling out and invoking pathmgr the same way a
user would.
"""

import sys
import os
import os.path
import inspect
from subprocess import Popen, PIPE

#
# This is the list of pathmgr tests to run.  These tests all shell out
# and invoke pathmgr the same way a user would and then verifies the
# output.
#
def pathmgr_tests():
  return [

    ########################################################################
    # Path Separator
    ########################################################################

    # Test
    dict(description='basic default path separator',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-l', '-q'],
         expected_result=0,
         expected_txtout=os.linesep.join(['foo', 'bar', 'baz', ''])),

    # Test
    dict(description='basic Unix path separator',
         path='foo:bar:baz',
         cmdline_params=['--sep=:', '-l', '-q'],
         expected_result=0,
         expected_txtout=os.linesep.join(['foo', 'bar', 'baz', ''])),

    # Test
    dict(description='basic Windows path separator',
         path='foo;bar;baz',
         cmdline_params=['--sep=;', '-l', '-q'],
         expected_result=0,
         expected_txtout=os.linesep.join(['foo', 'bar', 'baz', ''])),

    # Test
    dict(description='basic comma path separator',
         path='foo,bar,baz',
         cmdline_params=['--sep=,', '-l', '-q'],
         expected_result=0,
         expected_txtout=os.linesep.join(['foo', 'bar', 'baz', ''])),

    # Test
    dict(description='comma path separator, empty path',
         path=os.pathsep.join([]),
         cmdline_params=['--sep=,', '-l', '-q'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='comma path separator, single path',
         path='foo',
         cmdline_params=['--sep=,', '-l', '-q'],
         expected_result=0,
         expected_txtout=os.linesep.join(['foo', ''])),

    # Test
    dict(description='comma path separator, two empty path elements',
         path=',',
         cmdline_params=['--sep=,', '-l', '-q'],
         expected_result=0,
         expected_txtout=os.linesep.join(['', '', ''])),

    # Test
    dict(description='comma path separator, three empty path elements',
         path=',,',
         cmdline_params=['--sep=,', '-l', '-q'],
         expected_result=0,
         expected_txtout=os.linesep.join(['', '', '', ''])),

    ########################################################################
    # Origin
    ########################################################################

    dict(description='default environment variable (PATH) origin',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-l', '-q'],
         expected_result=0,
         expected_txtout=os.linesep.join(['foo', 'bar', 'baz', ''])),

    dict(description='non-default environment variable (CDPATH) origin ' \
                     'using --origin option.',
         env=dict(CDPATH=os.pathsep.join(['foo', 'bar', 'baz'])),
         cmdline_params=['--origin=CDPATH', '-l', '-q'],
         expected_result=0,
         expected_txtout=os.linesep.join(['foo', 'bar', 'baz', ''])),

    dict(description='non-default environment variable (CDPATH) origin ' \
                     'using --origin option.',
         env=dict(CDPATH=os.pathsep.join(['foo', 'bar', 'baz'])),
         cmdline_params=['--origin=CDPATH', '-l', '-q'],
         expected_result=0,
         expected_txtout=os.linesep.join(['foo', 'bar', 'baz', ''])),

    dict(description='standard input origin ' \
                     'using --origin option.',
         txtin=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['--origin=-', '-l', '-q'],
         expected_result=0,
         expected_txtout=os.linesep.join(['foo', 'bar', 'baz', ''])),

    ########################################################################
    # No Command Line Arguments
    ########################################################################

    # Test
    dict(description='empty path, no command line params',
         path=os.pathsep.join([]),
         cmdline_params=[],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='one path element, no command line params',
         path=os.pathsep.join(['foo']),
         cmdline_params=[],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='two path elements, no command line params',
         path=os.pathsep.join(['foo', 'bar']),
         cmdline_params=[],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='three path elements, no command line params',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=[],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz'])),

    ########################################################################
    # After / Append
    ########################################################################

    # Test
    dict(description='empty path, one append command using -a',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '-a=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='empty path, one append command using --after',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--after=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='empty path, one append command using --append',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--append=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='empty path, two append commands using -a',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '-a=foo', '-a=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='empty path, two append commands using --after',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--after=foo', '--after=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='empty path, two append commands using --append',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--append=foo', '--append=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='empty path, three append commands using -a',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '-a=foo', '-a=bar', '-a=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz'])),

    # Test
    dict(description='empty path, three append commands using --after',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off',
                         '--after=foo', '--after=bar', '--after=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz'])),

    # Test
    dict(description='empty path, three append commands using --append',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off',
                         '--append=foo', '--append=bar', '--append=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz'])),

    ########################################################################
    # Before / Prepend
    ########################################################################

    # Test
    dict(description='empty path, one prepend command using -b',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '-b=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='empty path, one prepend command using --before',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--before=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='empty path, one prepend command using --prepend',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--prepend=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='empty path, two prepend command using -b',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '-b=foo', '-b=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar', 'foo'])),

    # Test
    dict(description='empty path, two prepend command using --before',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--before=foo', '--before=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar', 'foo'])),

    # Test
    dict(description='empty path, two prepend command using --prepend',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--prepend=foo', '--prepend=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar', 'foo'])),

    # Test
    dict(description='empty path, three append commands using -b',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '-b=foo', '-b=bar', '-b=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['baz', 'bar', 'foo'])),

    # Test
    dict(description='empty path, three append commands using --before',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off',
                         '--before=foo', '--before=bar', '--before=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['baz', 'bar', 'foo'])),

    # Test
    dict(description='empty path, three append commands using --prepend',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off',
                         '--prepend=foo', '--prepend=bar', '--prepend=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['baz', 'bar', 'foo'])),

    # Test
    dict(description='one path element, ' \
                     'append and prepend using -a and -b',
         path=os.pathsep.join(['bar']),
         cmdline_params=['-f', '-n=off', '-a=baz', '-b=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz'])),

    # Test
    dict(description='one path element, ' \
                     'append and prepend using --after and --before',
         path=os.pathsep.join(['bar']),
         cmdline_params=['-f', '-n=off', '--after=baz', '--before=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz'])),

    # Test
    dict(description='one path element, ' \
                     'append and prepend using --append and --prepend',
         path=os.pathsep.join(['bar']),
         cmdline_params=['-f', '-n=off', '--append=baz', '--prepend=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz'])),

    # Test
    dict(description='one path element, ' \
                     'append and prepend using -b and -a',
         path=os.pathsep.join(['bar']),
         cmdline_params=['-f', '-n=off', '-b=foo', '-a=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz'])),

    # Test
    dict(description='one path element, ' \
                     'append and prepend using --before and --after',
         path=os.pathsep.join(['bar']),
         cmdline_params=['-f', '-n=off', '--before=foo', '--after=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz'])),

    # Test
    dict(description='one path element, ' \
                     'append and prepend using --prepend and --append',
         path=os.pathsep.join(['bar']),
         cmdline_params=['-f', '-n=off', '--prepend=foo', '--append=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz'])),

    ########################################################################
    # Delete
    ########################################################################

    # Test
    dict(description='empty path, one delete command using -d',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '-d=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='empty path, one delete command using --delete',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--delete=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='empty path, one delete command using --remove',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--remove=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='one path element, ' \
                     'one on-target delete command using -d',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-f', '-n=off', '-d=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='one path element, ' \
                     'one on-target delete command using --delete',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-f', '-n=off', '--delete=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='one path element, '\
                     'one on-target delete command using --remove',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-f', '-n=off', '--remove=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='one path element, ' \
                     'multiple off-target delete command using -d',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-f', '-n=off', '-d=bar', '-d=baz', '-d=spam'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='one path element, ' \
                     'multiple off-target delete command using --delete',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-f', '-n=off',
                         '--delete=bar', '--delete=baz', '--delete=spam'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='one path element, '\
                     'multiple off-target delete command using --remove',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-f', '-n=off',
                         '--remove=bar', '--remove=baz', '--remove=spam'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='two path elements, '\
                     'delete first element using -d',
         path=os.pathsep.join(['foo', 'bar']),
         cmdline_params=['-f', '-n=off', '-d=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar'])),

    # Test
    dict(description='two path elements, '\
                     'delete first element using --delete',
         path=os.pathsep.join(['foo', 'bar']),
         cmdline_params=['-f', '-n=off', '--delete=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar'])),

    # Test
    dict(description='two path elements, '\
                     'delete first element using --remove',
         path=os.pathsep.join(['foo', 'bar']),
         cmdline_params=['-f', '-n=off', '--remove=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar'])),

    # Test
    dict(description='two path elements, '\
                     'delete last element using -d',
         path=os.pathsep.join(['foo', 'bar']),
         cmdline_params=['-f', '-n=off', '-d=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='two path elements, '\
                     'delete last element using --delete',
         path=os.pathsep.join(['foo', 'bar']),
         cmdline_params=['-f', '-n=off', '--delete=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='two path elements, '\
                     'delete last element using --remove',
         path=os.pathsep.join(['foo', 'bar']),
         cmdline_params=['-f', '-n=off', '--remove=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='three path elements, '\
                     'delete first element using -d',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '-d=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar', 'baz'])),

    # Test
    dict(description='three path elements, '\
                     'delete first element using --delete',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--delete=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar', 'baz'])),

    # Test
    dict(description='three path elements, '\
                     'delete first element using --remove',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--remove=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar', 'baz'])),

    # Test
    dict(description='three path elements, '\
                     'delete middle element using -d',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '-d=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'baz'])),

    # Test
    dict(description='three path elements, '\
                     'delete middle element using --delete',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--delete=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'baz'])),

    # Test
    dict(description='three path elements, '\
                     'delete middle element using --remove',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--remove=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'baz'])),

    # Test
    dict(description='three path elements, '\
                     'delete last element using -d',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '-d=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='three path elements, '\
                     'delete last element using --delete',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--delete=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='three path elements, '\
                     'delete last element using --remove',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--remove=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='three path elements, '\
                     'delete identical first and last using one -d',
         path=os.pathsep.join(['foo', 'bar', 'foo']),
         cmdline_params=['-f', '-n=off', '-d=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar'])),

    # Test
    dict(description='three path elements, '\
                     'delete identical first and last using one --delete',
         path=os.pathsep.join(['foo', 'bar', 'foo']),
         cmdline_params=['-f', '-n=off', '--delete=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar'])),

    # Test
    dict(description='three path elements, '\
                     'delete identical first and last using one --remove',
         path=os.pathsep.join(['foo', 'bar', 'foo']),
         cmdline_params=['-f', '-n=off', '--remove=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar'])),

    # Test
    dict(description='three path elements, '\
                     'delete all using -d',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '-d=foo', '-d=bar', '-d=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='three path elements, '\
                     'delete all using --delete',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off',
                         '--delete=foo', '--delete=bar', '--delete=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='three path elements, '\
                     'delete all using --remove',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off',
                         '--remove=foo', '--remove=bar', '--remove=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    ########################################################################
    # Remove Duplicates
    ########################################################################

    # Test
    dict(description='empty path, -n=off, '\
                     'remove duplicates with --rmdups',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--rmdups'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='empty path, -n=off, '\
                     'remove duplicates with --remove-duplicates',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--remove-duplicates'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='empty path, -n=on, '\
                     'remove duplicates with --rmdups',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=on', '--rmdups'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='empty path, -n=on, '\
                     'remove duplicates with --remove-duplicates',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=on', '--remove-duplicates'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='one path element, -n=off, '\
                     'remove duplicates with --rmdups',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-f', '-n=off', '--rmdups'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='one path element, -n=off, '\
                     'remove duplicates with --remove-duplicates',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-f', '-n=off', '--remove-duplicates'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='one path element, -n=on, '\
                     'remove duplicates with --rmdups',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-f', '-n=on', '--rmdups'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='one path element, -n=on, '\
                     'remove duplicates with --remove-duplicates',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-f', '-n=on', '--remove-duplicates'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='two identical path elements, -n=off, '\
                     'remove duplicates with --rmdups',
         path=os.pathsep.join(['foo', 'foo']),
         cmdline_params=['-f', '-n=off', '--rmdups'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='two identical path elements, -n=off, '\
                     'remove duplicates with --remove-duplicates',
         path=os.pathsep.join(['foo', 'foo']),
         cmdline_params=['-f', '-n=off', '--remove-duplicates'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='two identical path elements, -n=on, '\
                     'remove duplicates with --rmdups',
         path=os.pathsep.join(['foo', 'foo']),
         cmdline_params=['-f', '-n=on', '--rmdups'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='two identical path elements, -n=on, '\
                     'remove duplicates with --remove-duplicates',
         path=os.pathsep.join(['foo', 'foo']),
         cmdline_params=['-f', '-n=on', '--remove-duplicates'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='three identical path elements, -n=off, '\
                     'remove duplicates with --rmdups',
         path=os.pathsep.join(['foo', 'foo', 'foo']),
         cmdline_params=['-f', '-n=off', '--rmdups'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='three identical path elements, -n=off, '\
                     'remove duplicates with --remove-duplicates',
         path=os.pathsep.join(['foo', 'foo', 'foo']),
         cmdline_params=['-f', '-n=off', '--remove-duplicates'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='three identical path elements, -n=on, '\
                     'remove duplicates with --rmdups',
         path=os.pathsep.join(['foo', 'foo', 'foo']),
         cmdline_params=['-f', '-n=on', '--rmdups'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='three identical path elements, -n=on, '\
                     'remove duplicates with --remove-duplicates',
         path=os.pathsep.join(['foo', 'foo', 'foo']),
         cmdline_params=['-f', '-n=on', '--remove-duplicates'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='leading identical path elements, -n=off, '\
                     'remove duplicates with --rmdups',
         path=os.pathsep.join(['foo', 'foo', 'bar']),
         cmdline_params=['-f', '-n=off', '--rmdups'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='leading identical path elements, -n=off, '\
                     'remove duplicates with --remove-duplicates',
         path=os.pathsep.join(['foo', 'foo', 'bar']),
         cmdline_params=['-f', '-n=off', '--remove-duplicates'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='leading identical path elements, -n=on, '\
                     'remove duplicates with --rmdups',
         path=os.pathsep.join(['foo', 'foo', 'bar']),
         cmdline_params=['-f', '-n=on', '--rmdups'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='leading identical path elements, -n=on, '\
                     'remove duplicates with --remove-duplicates',
         path=os.pathsep.join(['foo', 'foo', 'bar']),
         cmdline_params=['-f', '-n=on', '--remove-duplicates'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='trailing identical path elements, -n=off, '\
                     'remove duplicates with --rmdups',
         path=os.pathsep.join(['foo', 'bar', 'bar']),
         cmdline_params=['-f', '-n=off', '--rmdups'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='trailing identical path elements, -n=off, '\
                     'remove duplicates with --remove-duplicates',
         path=os.pathsep.join(['foo', 'bar', 'bar']),
         cmdline_params=['-f', '-n=off', '--remove-duplicates'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='trailing identical path elements, -n=on, '\
                     'remove duplicates with --rmdups',
         path=os.pathsep.join(['foo', 'bar', 'bar']),
         cmdline_params=['-f', '-n=on', '--rmdups'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='trailing identical path elements, -n=on, '\
                     'remove duplicates with --remove-duplicates',
         path=os.pathsep.join(['foo', 'bar', 'bar']),
         cmdline_params=['-f', '-n=on', '--remove-duplicates'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='split identical path elements, -n=off, '\
                     'remove duplicates with --rmdups',
         path=os.pathsep.join(['foo', 'bar', 'foo']),
         cmdline_params=['-f', '-n=off', '--rmdups'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='split identical path elements, -n=off, '\
                     'remove duplicates with --remove-duplicates',
         path=os.pathsep.join(['foo', 'bar', 'foo']),
         cmdline_params=['-f', '-n=off', '--remove-duplicates'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='split identical path elements, -n=on, '\
                     'remove duplicates with --rmdups',
         path=os.pathsep.join(['foo', 'bar', 'foo']),
         cmdline_params=['-f', '-n=on', '--rmdups'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='split identical path elements, -n=on, '\
                     'remove duplicates with --remove-duplicates',
         path=os.pathsep.join(['foo', 'bar', 'foo']),
         cmdline_params=['-f', '-n=on', '--remove-duplicates'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='identical path elements only when normalized, -n=off, '\
                     'remove duplicates with --rmdups',
         path=os.pathsep.join(['foo', 'tmp/../foo']),
         cmdline_params=['-f', '-n=off', '--rmdups'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'tmp/../foo'])),

    # Test
    dict(description='identical path elements only when normalized, -n=off, '\
                     'remove duplicates with --remove-duplicates',
         path=os.pathsep.join(['foo', 'tmp/../foo']),
         cmdline_params=['-f', '-n=off', '--remove-duplicates'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'tmp/../foo'])),

    # Test
    dict(description='identical path elements only when normalized, -n=on, '\
                     'remove duplicates with --rmdups',
         path=os.pathsep.join(['foo', 'tmp/../foo']),
         cmdline_params=['-f', '-n=on', '--rmdups'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='identical path elements only when normalized, -n=on, '\
                     'remove duplicates with --remove-duplicates',
         path=os.pathsep.join(['foo', 'tmp/../foo']),
         cmdline_params=['-f', '-n=on', '--remove-duplicates'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    ########################################################################
    # Clean
    ########################################################################

    # Test
    dict(description='empty path, clean nothing with -c',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '-c'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='empty path, clean nothing with --clean',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--clean'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='empty path, clean nothing with --clean-only',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--clean-only'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='one path element, '
                     'on-target clean command using -c',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-f', '-n=off', '-c', '-a=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='one path element, '
                     'on-target clean command using --clean',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-f', '-n=off', '--clean', '-a=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='one path element, '
                     'on-target clean command using --clean-only',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-f', '-n=off', '--clean-only', '-a=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='one path element, '
                     'clean with -c multiple off-target appends',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-f', '-n=off', '-c', '-a=bar', '-a=baz', '-a=spam'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='one path element, '
                     'clean with --clean multiple off-target appends',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-f', '-n=off', '--clean',
                         '-a=bar', '-a=baz', '-a=spam'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='one path element, '
                     'clean with --clean-only multiple off-target appends',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-f', '-n=off', '--clean-only',
                         '-a=bar', '-a=baz', '-a=spam'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='two path elements, '
                     'clean first element command using -c',
         path=os.pathsep.join(['foo', 'bar']),
         cmdline_params=['-f', '-n=off', '-c', '-a=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar'])),

    # Test
    dict(description='two path elements, '
                     'clean first element command using --clean',
         path=os.pathsep.join(['foo', 'bar']),
         cmdline_params=['-f', '-n=off', '--clean', '-a=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar'])),

    # Test
    dict(description='two path elements, '
                     'clean first element command using --clean-only',
         path=os.pathsep.join(['foo', 'bar']),
         cmdline_params=['-f', '-n=off', '--clean-only', '-a=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar'])),

    # Test
    dict(description='two path elements, '
                     'clean last element command using -c',
         path=os.pathsep.join(['foo', 'bar']),
         cmdline_params=['-f', '-n=off', '-c', '-a=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='two path elements, '
                     'clean last element command using --clean',
         path=os.pathsep.join(['foo', 'bar']),
         cmdline_params=['-f', '-n=off', '--clean', '-a=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='two path elements, '
                     'clean last element command using --clean-only',
         path=os.pathsep.join(['foo', 'bar']),
         cmdline_params=['-f', '-n=off', '--clean-only', '-a=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='three path elements, '
                     'clean first element command using -c',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '-c', '-a=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar' , 'baz'])),

    # Test
    dict(description='three path elements, '
                     'clean first element command using --clean',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--clean', '-a=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar' , 'baz'])),

    # Test
    dict(description='three path elements, '
                     'clean first element command using --clean-only',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--clean-only', '-a=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar' , 'baz'])),

    # Test
    dict(description='three path elements, '
                     'clean middle element command using -c',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '-c', '-a=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo' , 'baz'])),

    # Test
    dict(description='three path elements, '
                     'clean middle element command using --clean',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--clean', '-a=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo' , 'baz'])),

    # Test
    dict(description='three path elements, '
                     'clean middle element command using --clean-only',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--clean-only', '-a=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo' , 'baz'])),

    # Test
    dict(description='three path elements, '
                     'clean last element command using -c',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '-c', '-a=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo' , 'bar'])),

    # Test
    dict(description='three path elements, '
                     'clean last element command using --clean',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--clean', '-a=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo' , 'bar'])),

    # Test
    dict(description='three path elements, '
                     'clean last element command using --clean-only',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--clean-only', '-a=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo' , 'bar'])),

    # Test
    dict(description='three path elements, '
                     'clean identical first and last using one -c',
         path=os.pathsep.join(['foo', 'bar', 'foo']),
         cmdline_params=['-f', '-n=off', '-c', '-a=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar'])),

    # Test
    dict(description='three path elements, '
                     'clean identical first and last using one --clean',
         path=os.pathsep.join(['foo', 'bar', 'foo']),
         cmdline_params=['-f', '-n=off', '--clean', '-a=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar'])),

    # Test
    dict(description='three path elements, '
                     'clean identical first and last using one --clean-only',
         path=os.pathsep.join(['foo', 'bar', 'foo']),
         cmdline_params=['-f', '-n=off', '--clean-only', '-a=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar'])),

    # Test
    dict(description='three path elements, '\
                     'clean all using -c',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '-c', '-a=foo', '-a=bar', '-a=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='three path elements, '\
                     'clean all using --clean',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--clean',
                         '-a=foo', '-a=bar', '-a=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='three path elements, '\
                     'clean all using --clean-only',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--clean-only',
                         '-a=foo', '-a=bar', '-a=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    ########################################################################
    # Allow Duplicates
    ########################################################################

    # Test
    dict(description='allow non-contiguous duplicate, ' \
                     'using --dups and --after',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--dups', '--after=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz', 'bar'])),

    # Test
    dict(description='allow non-contiguous duplicate, ' \
                     'using --duplicates and --after',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--duplicates', '--after=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz', 'bar'])),

    # Test
    dict(description='allow non-contiguous duplicate, ' \
                     'using --allow-duplicates and --after',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--allow-duplicates', '--after=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz', 'bar'])),

    # Test
    dict(description='allow non-contiguous duplicate, ' \
                     'using --dups=on',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--dups=on', '--after=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz', 'bar'])),

    # Test
    dict(description='allow non-contiguous duplicate, ' \
                     'using --duplicates=on',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--duplicates=on', '--after=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz', 'bar'])),

    # Test
    dict(description='allow non-contiguous duplicate, ' \
                     'using --allow-duplicates=on',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off',
                         '--allow-duplicates=on', '--after=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz', 'bar'])),

    # Test
    dict(description='allow non-contiguous duplicate, ' \
                     'using --dups and --before',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--dups', '--before=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar', 'foo', 'bar', 'baz'])),

    # Test
    dict(description='allow non-contiguous duplicate, ' \
                     'using --duplicates and --before',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--duplicates', '--before=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar', 'foo', 'bar', 'baz'])),

    # Test
    dict(description='allow non-contiguous duplicate, ' \
                     'using --allow-duplicates and --before',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--allow-duplicates', '--before=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar', 'foo', 'bar', 'baz'])),

    # Test
    dict(description='allow non-contiguous duplicate, ' \
                     'using --dups=on',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--dups=on', '--before=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar', 'foo', 'bar', 'baz'])),

    # Test
    dict(description='allow non-contiguous duplicate, ' \
                     'using --duplicates=on',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--duplicates=on', '--before=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar', 'foo', 'bar', 'baz'])),

    # Test
    dict(description='allow non-contiguous duplicate, ' \
                     'using --allow-duplicates=on',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off',
                         '--allow-duplicates=on', '--before=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar', 'foo', 'bar', 'baz'])),

    # Test
    dict(description='disallow non-contiguous duplicate, ' \
                     'using --dups=off and --after',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--dups=off', '--after=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz'])),

    # Test
    dict(description='disallow non-contiguous duplicate, ' \
                     'using --duplicates=off and --after',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--duplicates=off', '--after=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz'])),

    # Test
    dict(description='disallow non-contiguous duplicate, ' \
                     'using --allow-duplicates=off and --after',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off',
                         '--allow-duplicates=off', '--after=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz'])),

    # Test
    dict(description='disallow non-contiguous duplicate, ' \
                     'using --dups=off and --before',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--dups=off', '--before=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz'])),

    # Test
    dict(description='disallow non-contiguous duplicate, ' \
                     'using --duplicates=off and --before',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--duplicates=off', '--before=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz'])),

    # Test
    dict(description='disallow non-contiguous duplicate, ' \
                     'using --allow-duplicates=off and --before',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off',
                         '--allow-duplicates=off', '--before=bar'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz'])),

    ########################################################################
    # Force - add even if missing from the file system
    ########################################################################

    # Test
    dict(description='add non-directory, ' \
                     'using -f and --after',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-n=off', '-f', '--univ-dir-seps=off',
                         '--after=/dir/not/present'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', '/dir/not/present'])),

    # Test
    dict(description='add non-directory, ' \
                     'using --force and --after',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-n=off', '--force', '--univ-dir-seps=off',
                         '--after=/dir/not/present'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', '/dir/not/present'])),

    # Test
    dict(description='add non-directory, ' \
                     'using -f and --before',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-n=off', '-f', '--univ-dir-seps=off',
                         '--before=/dir/not/present'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['/dir/not/present', 'foo'])),

    # Test
    dict(description='add non-directory, ' \
                     'using --force and --before',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-n=off', '--force', '--univ-dir-seps=off',
                         '--before=/dir/not/present'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['/dir/not/present', 'foo'])),

    # Test
    dict(description='add non-directory, ' \
                     'using -f=on and --after',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-n=off', '-f=on', '--univ-dir-seps=off',
                         '--after=/dir/not/present'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', '/dir/not/present'])),

    # Test
    dict(description='add non-directory, ' \
                     'using --force=on and --after',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-n=off', '--force=on', '--univ-dir-seps=off',
                         '--after=/dir/not/present'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', '/dir/not/present'])),

    # Test
    dict(description='add non-directory, ' \
                     'using -f=on and --before',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-n=off', '-f=on', '--univ-dir-seps=off',
                         '--before=/dir/not/present'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['/dir/not/present', 'foo'])),

    # Test
    dict(description='add non-directory, ' \
                     'using --force=on and --before',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-n=off', '--force=on', '--univ-dir-seps=off',
                         '--before=/dir/not/present'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['/dir/not/present', 'foo'])),

    # Test
    dict(description='do not add non-directory, ' \
                     'using -f=off and --after',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-n=off', '-f=off', '--univ-dir-seps=off',
                         '--after=/dir/not/present'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='do not add non-directory, ' \
                     'using --force=off and --after',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-n=off', '--force=off', '--univ-dir-seps=off',
                         '--after=/dir/not/present'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='do not add non-directory, ' \
                     'using -f=off and --before',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-n=off', '-f=off', '--univ-dir-seps=off',
                         '--before=/dir/not/present'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='do not add non-directory, ' \
                     'using --force=off and --before',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-n=off', '--force=off', '--univ-dir-seps=off',
                         '--before=/dir/not/present'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    ########################################################################
    # Normalize
    ########################################################################

    # Test
    dict(description='normalize path, ' \
                     'using -n and --after',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n', '--after=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.path.abspath('foo')])),

    # Test
    dict(description='normalize path, ' \
                     'using --normalize and --after',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '--normalize', '--after=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.path.abspath('foo')])),

    # Test
    dict(description='normalize path, ' \
                     'using -n=on and --after',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=on', '--after=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.path.abspath('foo')])),

    # Test
    dict(description='normalize path, ' \
                     'using --normalize=on and --after',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '--normalize=on', '--after=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.path.abspath('foo')])),

    # Test
    dict(description='do not normalize path, ' \
                     'using -n=off and --after',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--after=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='do not normalize path, ' \
                     'using --normalize=off and --after',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '--normalize=off', '--after=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    ########################################################################
    # Universal Directory Separators
    ########################################################################

    # Test
    dict(description='universal directory separators off, '
                     'using --uds',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--dups=on', '--uds=off',
                         '--after=foo/bar/baz', '--after=foo\\bar\\baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo/bar/baz', 'foo\\bar\\baz'])),

    # Test
    dict(description='universal directory separators off, '
                     'using --univ-dir-seps',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--dups=on', '--univ-dir-seps=off',
                         '--after=foo/bar/baz', '--after=foo\\bar\\baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo/bar/baz', 'foo\\bar\\baz'])),

    # Test
    dict(description='universal directory separators off, '
                     'using --universal-directory-separators',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--dups=on',
                         '--universal-directory-separators=off',
                         '--after=foo/bar/baz', '--after=foo\\bar\\baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo/bar/baz', 'foo\\bar\\baz'])),

    # Test
    dict(description='universal directory separators on, '
                     'using --uds',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--dups=on', '--uds=on',
                         '--after=foo/bar/baz', '--after=foo\\bar\\baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.sep.join(['foo', 'bar', 'baz']),
                                          os.sep.join(['foo', 'bar', 'baz'])])),

    # Test
    dict(description='universal directory separators on, '
                     'using --univ-dir-seps',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--dups=on', '--univ-dir-seps=on',
                         '--after=foo/bar/baz', '--after=foo\\bar\\baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.sep.join(['foo', 'bar', 'baz']),
                                          os.sep.join(['foo', 'bar', 'baz'])])),

    # Test
    dict(description='universal directory separators on, '
                     'using --universal-directory-separators',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--dups=on',
                         '--universal-directory-separators=on',
                         '--after=foo/bar/baz', '--after=foo\\bar\\baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.sep.join(['foo', 'bar', 'baz']),
                                          os.sep.join(['foo', 'bar', 'baz'])])),

    ########################################################################
    # Tilde Expansion
    ########################################################################

    # Test
    dict(description='tilde expansion off, using --tilde=off',
         env=dict(HOME='foo'),
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--dups=on', '--tilde=off',
                         '--after=~/bar/baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.sep.join(['~', 'bar', 'baz'])])),

    # Test
    dict(description='tilde expansion off, using --tilde-expansion=off',
         env=dict(HOME='foo'),
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--dups=on', '--tilde-expansion=off',
                         '--after=~/bar/baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.sep.join(['~', 'bar', 'baz'])])),

    # Test
    dict(description='tilde expansion on, using --tilde',
         env=dict(HOME='foo'),
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--dups=on', '--tilde',
                         '--after=~/bar/baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.sep.join(['foo', 'bar', 'baz'])])),

    # Test
    dict(description='tilde expansion on, using --tilde-expansion',
         env=dict(HOME='foo'),
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--dups=on', '--tilde-expansion',
                         '--after=~/bar/baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.sep.join(['foo', 'bar', 'baz'])])),

    # Test
    dict(description='tilde expansion on, using --tilde=on',
         env=dict(HOME='foo'),
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--dups=on', '--tilde=on',
                         '--after=~/bar/baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.sep.join(['foo', 'bar', 'baz'])])),

    # Test
    dict(description='tilde expansion on, using --tilde-expansion=on',
         env=dict(HOME='foo'),
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--dups=on', '--tilde-expansion=on',
                         '--after=~/bar/baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.sep.join(['foo', 'bar', 'baz'])])),

    ########################################################################
    # Anchor
    ########################################################################

    # Test
    dict(description='one element with empty anchor',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--anchor=', '--after=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo'])),

    # Test
    dict(description='one element with non-empty anchor',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--anchor=..', '--after=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.sep.join(['..', 'foo'])])),

    # Test
    dict(description='three elements with non-empty anchor',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--anchor=..',
                         '-a=foo', '-a=bar', '-a=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.sep.join(['..', 'foo']),
                                          os.sep.join(['..', 'bar']),
                                          os.sep.join(['..', 'baz'])])),

    # Test
    dict(description='three elements each with different non-empty anchors',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off',
                         '--anchor=%s' % (os.sep.join(['..'])),
                         '-a=foo',
                         '--anchor=%s' % (os.sep.join(['..', '..'])),
                         '-a=bar',
                         '--anchor=%s' % (os.sep.join(['..', '..', '..'])),
                         '-a=baz'],
         expected_result=0,
         expected_txtout=os.pathsep.join(
             [os.sep.join(['..', 'foo']),
              os.sep.join(['..', '..', 'bar']),
              os.sep.join(['..', '..', '..', 'baz'])])),

    ########################################################################
    # Default / Define / Undefine
    ########################################################################

    # Test
    dict(description='default value',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--default=foo=bar', '-a=${foo}'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar'])),

    # Test
    dict(description='define value, using -D=',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '-D=foo=bar', '-a=${foo}'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar'])),

    # Test
    dict(description='define value, using --def=',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--def=foo=bar', '-a=${foo}'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar'])),

    # Test
    dict(description='define value, using --define=',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--define=foo=bar', '-a=${foo}'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar'])),

    # Test
    dict(description='undefine value, using -U=',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--define=foo=bar', '-U=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='undefine value, using --undef=',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--define=foo=bar', '--undef=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='undefine value, using --undefine=',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--define=foo=bar', '--undefine=foo'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='undefine an undefined value, using -U=',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '-U=foo'],
         expected_result=1,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='twice undefine a defined value, using -U=',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--define=foo=bar',
                         '-U=foo', '-U=foo'],
         expected_result=1,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='override value',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off',
                         '--default=foo=bar',
                         '-D=foo=spam', '-a=${foo}'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['spam'])),

    # Test
    dict(description='undefine overridden value',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--dups=on',
                         '--default=foo=bar', '-a=${foo}',
                         '-D=foo=spam', '-a=${foo}',
                         '-U=foo', '-a=${foo}'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar', 'spam', 'bar'])),

    # Test
    dict(description='redefine overridden value',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--dups=on',
                         '--default=foo=bar', '-a=${foo}',
                         '-D=foo=spam', '-a=${foo}',
                         '-D=foo=eggs', '-a=${foo}'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['bar', 'spam', 'eggs'])),

    ########################################################################
    # Print State
    ########################################################################

    # Test
    dict(description='print state off',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '--print-state=off'],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'baz'])),

    ########################################################################
    # XML Configuration File
    ########################################################################

    # Test
    dict(description='pathmgr_test_missing.xml: missing',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', 'pathmgr_test_missing.xml'],
         expected_result=1,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='pathmgr_test_0001.xml: empty',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off',
                         os.path.join(thisdir(), 'pathmgr_test_0001.xml')],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='pathmgr_test_0002.xml: locally scoped anchor ' \
                     'using default for file name',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--dups=on',
                         os.path.join(thisdir(), 'pathmgr_test_0002.xml'),
                         '-a=spam', '-a=eggs'],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.sep.join(['foo', 'spam']),
                                          os.sep.join(['foo', 'eggs']),
                                          os.sep.join(['spam']),
                                          os.sep.join(['eggs'])])),

    # Test
    dict(description='pathmgr_test_000{1,2}.xml: two xml files',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--dups=on',
                         os.path.join(thisdir(), 'pathmgr_test_0001.xml'),
                         os.path.join(thisdir(), 'pathmgr_test_0002.xml')],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.sep.join(['foo', 'spam']),
                                          os.sep.join(['foo', 'eggs']),])),

    # Test
    dict(description='pathmgr_test_000{1,2,1}.xml: three xml files',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--dups=on',
                         os.path.join(thisdir(), 'pathmgr_test_0001.xml'),
                         os.path.join(thisdir(), 'pathmgr_test_0002.xml'),
                         os.path.join(thisdir(), 'pathmgr_test_0001.xml')],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.sep.join(['foo', 'spam']),
                                          os.sep.join(['foo', 'eggs']),])),

    # Test
    dict(description='pathmgr_test_0002.xml: locally scoped anchor ' \
                     'using --include',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--dups=on',
                         '--include=%s'
                         % (os.path.join(thisdir(), 'pathmgr_test_0002.xml'),),
                         '-a=spam', '-a=eggs'],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.sep.join(['foo', 'spam']),
                                          os.sep.join(['foo', 'eggs']),
                                          os.sep.join(['spam']),
                                          os.sep.join(['eggs'])])),

    # Test
    dict(description='pathmgr_test_0002.xml: globally scoped anchor',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--dups=on',
                         '--source=%s'
                         % (os.path.join(thisdir(), 'pathmgr_test_0002.xml'),),
                         '-a=spam', '-a=eggs'],
         expected_result=0,
         expected_txtout=os.pathsep.join([os.sep.join(['foo', 'spam']),
                                          os.sep.join(['foo', 'eggs']),
                                          os.sep.join(['foo', 'spam']),
                                          os.sep.join(['foo', 'eggs'])])),

    # Test
    dict(description='pathmgr_test_0002.xml: clean only',
         path=os.pathsep.join([os.sep.join(['foo', 'spam']),
                               os.sep.join(['foo', 'eggs']),
                               os.sep.join(['spam']),
                               os.sep.join(['eggs'])]),
         cmdline_params=['-f', '-n=off', '--dups=on', '--clean-only',
                         os.path.join(thisdir(), 'pathmgr_test_0002.xml'),
                         '-a=spam', '-a=eggs'],
         expected_result=0,
         expected_txtout=os.pathsep.join([])),

    # Test
    dict(description='pathmgr_test_0003.xml: portable directory separators',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--univ-dir-seps=on',
                         os.path.join(thisdir(), 'pathmgr_test_0003.xml')],
         expected_result=0,
         expected_txtout=os.pathsep.join([
             os.sep.join(['foo', 'bar', 'baz']),
             os.sep.join(['spam', 'eggs', 'quux']),
             'foo/bar/baz',
             'spam\\eggs\\quux',
             os.sep.join(['foo', 'bar', 'baz']),
             os.sep.join(['spam', 'eggs', 'quux']),
             'foo/bar/baz',
             'spam\\eggs\\quux'
         ])),

    # Test
    dict(description='pathmgr_test_0004.xml: tilde expansion',
         env=dict(HOME='foo'),
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off', '--tilde=on',
                         os.path.join(thisdir(), 'pathmgr_test_0004.xml')],
         expected_result=0,
         expected_txtout=os.pathsep.join([
              # With --univ-dir-seps=on.
              os.sep.join(['foo', 'bar',  'baz']),
              os.sep.join(['foo', 'eggs', 'quux']),
              os.sep.join(['~',   'bar',  'baz']),
              os.sep.join(['~',   'eggs', 'quux']),
              os.sep.join(['foo', 'bar',  'baz']),
              os.sep.join(['foo', 'eggs', 'quux']),
              os.sep.join(['~',   'bar',  'baz']),
              os.sep.join(['~',   'eggs', 'quux']),
              # With --univ-dir-seps=off.
              lite_os_specific_tilde_expand('~/bar/baz', home='foo'),
              lite_os_specific_tilde_expand('~\\eggs\\quux', home='foo'),
              '~/bar/baz',
              '~\\eggs\\quux',
              lite_os_specific_tilde_expand('~/bar/baz', home='foo'),
              lite_os_specific_tilde_expand('~\\eggs\\quux', home='foo'),
              '~/bar/baz',
              '~\\eggs\\quux',
         ])),

    # Test
    dict(description='pathmgr_test_0005.xml: after, append, before, prepend',
         path=os.pathsep.join(['foo']),
         cmdline_params=['-f', '-n=off',
                         os.path.join(thisdir(), 'pathmgr_test_0005.xml')],
         expected_result=0,
         expected_txtout=os.pathsep.join(['eggs', 'spam',
                                          'foo', 'bar', 'baz'])),

    # Test
    dict(description='pathmgr_test_0006.xml: no <separator> tag allowed',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off',
                         os.path.join(thisdir(), 'pathmgr_test_0006.xml')],
         expected_result=1,
         expected_txtout=''),

    # Test
    dict(description='pathmgr_test_0007.xml: no <origin> tag allowed',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off',
                         os.path.join(thisdir(), 'pathmgr_test_0007.xml')],
         expected_result=1,
         expected_txtout=''),

    # Test
    dict(description='pathmgr_test_0008.xml: no <log-to-stderr> tag allowed',
         path=os.pathsep.join([]),
         cmdline_params=['-f', '-n=off',
                         os.path.join(thisdir(), 'pathmgr_test_0008.xml')],
         expected_result=1,
         expected_txtout=''),

    # Test
    dict(description='pathmgr_test_0009.xml: list',
         path=os.pathsep.join(['foo', 'bar', 'baz']),
         cmdline_params=['-f', '-n=off', '-q',
                         os.path.join(thisdir(), 'pathmgr_test_0009.xml')],
         expected_result=0,
         expected_txtout=os.linesep.join([
           'foo', 'bar', 'baz', '-----', 'foo', 'bar', 'baz', ''])),

    # Test
    dict(description='pathmgr_test_0010.xml: remove duplicates ' \
                     'for path elements that are identical only ' \
                     'when normalized, -n=off',
         path=os.pathsep.join(['foo', 'bar', 'tmp/../foo']),
         cmdline_params=['-f', '-n=off',
                         os.path.join(thisdir(), 'pathmgr_test_0010.xml')],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'tmp/../foo'])),

    # Test
    dict(description='pathmgr_test_0010.xml: remove duplicates ' \
                     'for path elements that are identical only ' \
                     'when normalized, -n=on',
         path=os.pathsep.join(['foo', 'bar', 'tmp/../foo']),
         cmdline_params=['-f', '-n=on',
                         os.path.join(thisdir(), 'pathmgr_test_0010.xml')],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

    # Test
    dict(description='pathmgr_test_0011.xml: remove duplicates ' \
                     'for path elements that are identical only ' \
                     'when normalized, -n=off',
         path=os.pathsep.join(['foo', 'bar', 'tmp/../foo']),
         cmdline_params=['-f', '-n=off',
                         os.path.join(thisdir(), 'pathmgr_test_0011.xml')],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar', 'tmp/../foo'])),

    # Test
    dict(description='pathmgr_test_0011.xml: remove duplicates ' \
                     'for path elements that are identical only ' \
                     'when normalized, -n=on',
         path=os.pathsep.join(['foo', 'bar', 'tmp/../foo']),
         cmdline_params=['-f', '-n=on',
                         os.path.join(thisdir(), 'pathmgr_test_0011.xml')],
         expected_result=0,
         expected_txtout=os.pathsep.join(['foo', 'bar'])),

  ]

########################################################################
# Code for Running Tests
########################################################################

def thisdir():
    """ Return this directory."""
    return os.path.dirname(inspect.getabsfile(thisdir))

def lite_os_specific_tilde_expand(dir, home):
    """Tricky because, under Linux, "~\\foo" is treated as "~user" where
    user="\\foo" which is not a valid user which suppresses tilde
    expansion, but under Windows both os.sep and os.altsep are set so
    both "~/foo" and "~\\foo" expand.  We also cannot just call
    os.path.expanduser() directly because our $HOME is probably not
    set to "foo" as in the test."""
    for sep in (os.sep, os.altsep):
      parts = dir.split(sep)
      if parts and (parts[0] == '~'):
        parts[0] = home
        return sep.join(parts)
    return dir

def run(path=None, env=None, txtin=None, cmdline_params=None, **kwrest):
    """Test pathmgr by invoking it with the list of command-line parameters
    test_params with the environment variables defined in the env hash.

    If path is not None, it overrides the PATH environment variable in
    hash.  For most tests, path is defined and env is None which results
    in an empty environment except the PATH environment variable will be
    set to path.

    If txtin is not None, it is the text passed to pathmgr on its
    standard input.

    This function returns the tuple (returncode, txtout, txterr) where
    returncode is the value returned by pathmgr to the shell, txtout is
    the text written by pathmger to standard output, and txterr is the
    text writen by pathmgr to standard error."""

    # cmdline_params should default to an empty list.  This is how it is
    # done in Python.  You set the default to None and then check in the
    # body of your function to set it to an empty list; otherwise,
    # mutations of the default list will be picked up automatically the
    # next time this function is called because Python binds the default
    # value at compile-time only.
    if cmdline_params is None:
        cmdline_params = []

    # Prepend the "python" executable and "pathmgr.py" script to the
    # test_params.  Copy cmdline_params so we modify our copy, not the
    # user's.
    pathmgr = os.path.join(thisdir(), '..', 'pathmgr.py')
    test_params = list(cmdline_params)
    test_params[:0] = [sys.executable, pathmgr]

    # Set up the kwargs used for configuring our Popen() object.  If no
    # hash of environment variables is passed in via cmdline_params, an
    # empty one is created with PATH=path if path is not None.
    test_kwargs = {}
    test_kwargs['stdin'] = PIPE
    test_kwargs['stdout'] = PIPE
    test_kwargs['stderr'] = PIPE
    if env is not None:
        test_kwargs['env'] = env
    else:
        test_kwargs['env'] = dict()
    if path is not None:
        test_kwargs['env']['PATH'] = path

    # Shell out.  We have to handle the special case where Python-3.x
    # needs to have the text written to pathmgr's standard input encoded.
    proc = Popen(test_params, **test_kwargs)
    if (sys.version_info[0] >= 3) and (txtin is not None):
        (txtout, txterr) = proc.communicate(txtin.encode())
    else:
        (txtout, txterr) = proc.communicate(txtin)

    # Return results.
    return (proc.returncode, txtout, txterr)

def print_failure(test, actual_results):
    """Print the failure message for test and the actual results."""
    sys.stdout.write('-----\n')
    sys.stdout.write('Failed:\n')
    sys.stdout.write('%20s: %s\n' % ('description', test.get('description')))
    sys.stdout.write('%20s: %s\n' % ('env', test.get('env')))
    sys.stdout.write('%20s: %s\n' % ('PATH', test.get('path')))
    sys.stdout.write('%20s: %s\n' % ('txtin', test.get('txtin')))
    sys.stdout.write('%20s: %s\n' % ('params', test.get('cmdline_params')))
    sys.stdout.write('%20s: %s\n' % ('result', test.get('expected_result')))
    sys.stdout.write('%20s: %s\n' % ('result', actual_results[0],))
    sys.stdout.write('%20s: %s\n' % ('expected txtout',
                                     test.get('expected_txtout')))
    sys.stdout.write('%20s: %s\n' % ('actual txtout',
                                     actual_results[1],))
    sys.stdout.write('%20s: %s\n' % ('actual txterr',
                                     actual_results[2],))

def actual_matches_expected(test, actual_results):
    """Return True if the actual results match the expected results for the
    test."""

    # Create the tuple for the expected results.
    expected_results = (test['expected_result'],
                        test['expected_txtout'])

    # Python-3.x requires decoding the strings returned by pathmgr from
    # UTF-8 to the default encoding.
    if sys.version_info[0] >= 3:
        actual_results = (actual_results[0],            # return code
                          actual_results[1].decode())   # text on stdout

    return actual_results[:2] == expected_results

def main(quiet=False):
    total = 0
    passed = 0
    failed = 0

    # Iterate over each test.
    for test in pathmgr_tests():

        # Increase the test count.
        total += 1

        # Run the test.
        actual_results = run(**test)

        # Check the results for the test.
        if actual_matches_expected(test, actual_results):
            passed += 1
            if not quiet:
              sys.stdout.write('passed: ' + test['description'] + '\n')
        else:
            failed += 1
            print_failure(test, actual_results)

    # Print the final results.
    sys.stdout.write('-----\n')
    sys.stdout.write('failed=%d   passed=%d   total=%d\n'
                     % (failed, passed, total))
    sys.stdout.write('-----\n')

if __name__ == '__main__':

    # Check for quiet operation.
    quiet = False
    for arg in sys.argv:
      if arg in ['-q', '-quiet', '--quiet']:
        quiet = True
        break

    main(quiet)
