(defpackage :rational-to-string-test
  (:use :cl :rational-to-string :lisp-unit)
  (:export #:main))

(in-package :rational-to-string-test)

(defun harmonic-series (n)
  "Calculates the sum of the first N terms of the harmonic series:
    1 + 1/2 + 1/3 + ... + 1/n"
  (labels
      ((iter (n acc)
         (cond ((<= n 0) acc)
               (t (iter (- n 1) (+ (/ n) acc))))))
    (iter n 0)))

;;;
;;; (rational-to-string)
;;;

(define-test normalize-test

  ;;
  ;; x = 0
  ;;
  (assert-equal (values 0  0)             (normalize-rational  0))

  ;;
  ;; x = 1
  ;;
  (assert-equal (values 1  0)             (normalize-rational  1))

  ;;
  ;; abs(x) < 1
  ;;
  (assert-equal (values 7 -3)             (normalize-rational (/  7 1000)))
  (assert-equal (values 8 -3)             (normalize-rational (/  8 1000)))
  (assert-equal (values 9 -3)             (normalize-rational (/  9 1000)))
  (assert-equal (values 1 -2)             (normalize-rational (/ 10 1000)))
  (assert-equal (values (/ 11 10) -2)     (normalize-rational (/ 11 1000)))
  (assert-equal (values (/ 12 10) -2)     (normalize-rational (/ 12 1000)))
  (assert-equal (values (/ 13 10) -2)     (normalize-rational (/ 13 1000)))

  (assert-equal (values 7 -2)             (normalize-rational (/  7 100)))
  (assert-equal (values 8 -2)             (normalize-rational (/  8 100)))
  (assert-equal (values 9 -2)             (normalize-rational (/  9 100)))
  (assert-equal (values 1 -1)             (normalize-rational (/ 10 100)))
  (assert-equal (values (/ 11 10) -1)     (normalize-rational (/ 11 100)))
  (assert-equal (values (/ 12 10) -1)     (normalize-rational (/ 12 100)))
  (assert-equal (values (/ 13 10) -1)     (normalize-rational (/ 13 100)))

  (assert-equal (values 7 -1)             (normalize-rational (/  7 10)))
  (assert-equal (values 8 -1)             (normalize-rational (/  8 10)))
  (assert-equal (values 9 -1)             (normalize-rational (/  9 10)))
  (assert-equal (values 1  0)             (normalize-rational (/ 10 10)))
  (assert-equal (values (/ 11 10) 0)      (normalize-rational (/ 11 10)))
  (assert-equal (values (/ 12 10) 0)      (normalize-rational (/ 12 10)))
  (assert-equal (values (/ 13 10) 0)      (normalize-rational (/ 13 10)))

  (assert-equal (values -7 -3)            (normalize-rational (/  -7 1000)))
  (assert-equal (values -8 -3)            (normalize-rational (/  -8 1000)))
  (assert-equal (values -9 -3)            (normalize-rational (/  -9 1000)))
  (assert-equal (values -1 -2)            (normalize-rational (/ -10 1000)))
  (assert-equal (values (/ -11 10) -2)    (normalize-rational (/ -11 1000)))
  (assert-equal (values (/ -12 10) -2)    (normalize-rational (/ -12 1000)))
  (assert-equal (values (/ -13 10) -2)    (normalize-rational (/ -13 1000)))

  (assert-equal (values -7 -2)            (normalize-rational (/  -7 100)))
  (assert-equal (values -8 -2)            (normalize-rational (/  -8 100)))
  (assert-equal (values -9 -2)            (normalize-rational (/  -9 100)))
  (assert-equal (values -1 -1)            (normalize-rational (/ -10 100)))
  (assert-equal (values (/ -11 10) -1)    (normalize-rational (/ -11 100)))
  (assert-equal (values (/ -12 10) -1)    (normalize-rational (/ -12 100)))
  (assert-equal (values (/ -13 10) -1)    (normalize-rational (/ -13 100)))

  (assert-equal (values -7 -1)            (normalize-rational (/  -7 10)))
  (assert-equal (values -8 -1)            (normalize-rational (/  -8 10)))
  (assert-equal (values -9 -1)            (normalize-rational (/  -9 10)))
  (assert-equal (values -1  0)            (normalize-rational (/ -10 10)))
  (assert-equal (values (/ -11 10) 0)     (normalize-rational (/ -11 10)))
  (assert-equal (values (/ -12 10) 0)     (normalize-rational (/ -12 10)))
  (assert-equal (values (/ -13 10) 0)     (normalize-rational (/ -13 10)))

  ;;
  ;; abs(x) > 1
  ;;
  (assert-equal (values 7 0)              (normalize-rational  7))
  (assert-equal (values 8 0)              (normalize-rational  8))
  (assert-equal (values 9 0)              (normalize-rational  9))
  (assert-equal (values 1 1)              (normalize-rational 10))
  (assert-equal (values (/ 11 10) 1)      (normalize-rational 11))
  (assert-equal (values (/ 12 10) 1)      (normalize-rational 12))
  (assert-equal (values (/ 13 10) 1)      (normalize-rational 13))

  (assert-equal (values (/ 97 10) 1)      (normalize-rational  97))
  (assert-equal (values (/ 98 10) 1)      (normalize-rational  98))
  (assert-equal (values (/ 99 10) 1)      (normalize-rational  99))
  (assert-equal (values 1 2)              (normalize-rational 100))
  (assert-equal (values (/ 101 100) 2)    (normalize-rational 101))
  (assert-equal (values (/ 102 100) 2)    (normalize-rational 102))
  (assert-equal (values (/ 103 100) 2)    (normalize-rational 103))

  (assert-equal (values (/ 997 100) 2)    (normalize-rational  997))
  (assert-equal (values (/ 998 100) 2)    (normalize-rational  998))
  (assert-equal (values (/ 999 100) 2)    (normalize-rational  999))
  (assert-equal (values 1 3)              (normalize-rational 1000))
  (assert-equal (values (/ 1001 1000) 3)  (normalize-rational 1001))
  (assert-equal (values (/ 1002 1000) 3)  (normalize-rational 1002))
  (assert-equal (values (/ 1003 1000) 3)  (normalize-rational 1003))

  (assert-equal (values -7 0)             (normalize-rational  -7))
  (assert-equal (values -8 0)             (normalize-rational  -8))
  (assert-equal (values -9 0)             (normalize-rational  -9))
  (assert-equal (values -1 1)             (normalize-rational -10))
  (assert-equal (values (/ -11 10) 1)     (normalize-rational -11))
  (assert-equal (values (/ -12 10) 1)     (normalize-rational -12))
  (assert-equal (values (/ -13 10) 1)     (normalize-rational -13))

  (assert-equal (values (/ -97 10) 1)     (normalize-rational  -97))
  (assert-equal (values (/ -98 10) 1)     (normalize-rational  -98))
  (assert-equal (values (/ -99 10) 1)     (normalize-rational  -99))
  (assert-equal (values -1 2)             (normalize-rational -100))
  (assert-equal (values (/ -101 100) 2)   (normalize-rational -101))
  (assert-equal (values (/ -102 100) 2)   (normalize-rational -102))
  (assert-equal (values (/ -103 100) 2)   (normalize-rational -103))

  (assert-equal (values (/ -997 100) 2)   (normalize-rational  -997))
  (assert-equal (values (/ -998 100) 2)   (normalize-rational  -998))
  (assert-equal (values (/ -999 100) 2)   (normalize-rational  -999))
  (assert-equal (values -1 3)             (normalize-rational -1000))
  (assert-equal (values (/ -1001 1000) 3) (normalize-rational -1001))
  (assert-equal (values (/ -1002 1000) 3) (normalize-rational -1002))
  (assert-equal (values (/ -1003 1000) 3) (normalize-rational -1003)))

(define-test negative-test
  (assert-error 'error    (rational-to-string -1 -1 :trim t))

  (assert-equal "-1"      (rational-to-string -1 0 :trim t))
  (assert-equal "-1"      (rational-to-string -1 1 :trim t))
  (assert-equal "-1"      (rational-to-string -1 2 :trim t))
  (assert-equal "-1"      (rational-to-string -1 3 :trim t))
  (assert-equal "-1"      (rational-to-string -1 4 :trim t))

  (assert-equal "-1"      (rational-to-string -1 0 :trim nil))
  (assert-equal "-1.0"    (rational-to-string -1 1 :trim nil))
  (assert-equal "-1.00"   (rational-to-string -1 2 :trim nil))
  (assert-equal "-1.000"  (rational-to-string -1 3 :trim nil))
  (assert-equal "-1.0000" (rational-to-string -1 4 :trim nil))

  (assert-equal "-1"      (rational-to-string (/ -10001 10000) 0 :trim t))
  (assert-equal "-1"      (rational-to-string (/ -10001 10000) 1 :trim t))
  (assert-equal "-1"      (rational-to-string (/ -10001 10000) 2 :trim t))
  (assert-equal "-1"      (rational-to-string (/ -10001 10000) 3 :trim t))
  (assert-equal "-1.0001" (rational-to-string (/ -10001 10000) 4 :trim t))

  (assert-equal "-1"      (rational-to-string (/ -10001 10000) 0 :trim nil))
  (assert-equal "-1.0"    (rational-to-string (/ -10001 10000) 1 :trim nil))
  (assert-equal "-1.00"   (rational-to-string (/ -10001 10000) 2 :trim nil))
  (assert-equal "-1.000"  (rational-to-string (/ -10001 10000) 3 :trim nil))
  (assert-equal "-1.0001" (rational-to-string (/ -10001 10000) 4 :trim nil)))

(define-test zero-test
  (assert-error 'error  (rational-to-string 0 -1 :trim t))
  (assert-equal "0"     (rational-to-string 0  0 :trim t))
  (assert-equal "0"     (rational-to-string 0  1 :trim t))
  (assert-equal "0"     (rational-to-string 0  2 :trim t))
  (assert-equal "0"     (rational-to-string 0  3 :trim t))
  (assert-equal "0"     (rational-to-string 0  0 :trim nil))
  (assert-equal "0.0"   (rational-to-string 0  1 :trim nil))
  (assert-equal "0.00"  (rational-to-string 0  2 :trim nil))
  (assert-equal "0.000" (rational-to-string 0  3 :trim nil)))

(define-test positive-test
  (assert-error 'error   (rational-to-string -1 -1 :trim t))

  (assert-equal "1"      (rational-to-string 1 0 :trim t))
  (assert-equal "1"      (rational-to-string 1 1 :trim t))
  (assert-equal "1"      (rational-to-string 1 2 :trim t))
  (assert-equal "1"      (rational-to-string 1 3 :trim t))
  (assert-equal "1"      (rational-to-string 1 4 :trim t))

  (assert-equal "1"      (rational-to-string 1 0 :trim nil))
  (assert-equal "1.0"    (rational-to-string 1 1 :trim nil))
  (assert-equal "1.00"   (rational-to-string 1 2 :trim nil))
  (assert-equal "1.000"  (rational-to-string 1 3 :trim nil))
  (assert-equal "1.0000" (rational-to-string 1 4 :trim nil))

  (assert-equal "1"      (rational-to-string (/ 10001 10000) 0 :trim nil))
  (assert-equal "1.0"    (rational-to-string (/ 10001 10000) 1 :trim nil))
  (assert-equal "1.00"   (rational-to-string (/ 10001 10000) 2 :trim nil))
  (assert-equal "1.000"  (rational-to-string (/ 10001 10000) 3 :trim nil))
  (assert-equal "1.0001" (rational-to-string (/ 10001 10000) 4 :trim nil)))

(define-test no-trim-test
  (assert-equal "1"       (rational-to-string (/ 1001 1000) 0 :trim nil))
  (assert-equal "1.0"     (rational-to-string (/ 1001 1000) 1 :trim nil))
  (assert-equal "1.00"    (rational-to-string (/ 1001 1000) 2 :trim nil))
  (assert-equal "1.001"   (rational-to-string (/ 1001 1000) 3 :trim nil))
  (assert-equal "1.0010"  (rational-to-string (/ 1001 1000) 4 :trim nil))
  (assert-equal "1.00100" (rational-to-string (/ 1001 1000) 5 :trim nil)))

(define-test trim-test
  (assert-equal "1"     (rational-to-string (/ 1001 1000) 0 :trim t))
  (assert-equal "1"     (rational-to-string (/ 1001 1000) 1 :trim t))
  (assert-equal "1"     (rational-to-string (/ 1001 1000) 2 :trim t))
  (assert-equal "1.001" (rational-to-string (/ 1001 1000) 3 :trim t))
  (assert-equal "1.001" (rational-to-string (/ 1001 1000) 4 :trim t))
  (assert-equal "1.001" (rational-to-string (/ 1001 1000) 5 :trim t)))

(define-test rounding-test
  (assert-equal "0"       (rational-to-string (/ 1 100000) 0 :trim t))
  (assert-equal "0"       (rational-to-string (/ 1 100000) 1 :trim t))
  (assert-equal "0"       (rational-to-string (/ 1 100000) 2 :trim t))
  (assert-equal "0"       (rational-to-string (/ 1 100000) 3 :trim t))
  (assert-equal "0"       (rational-to-string (/ 1 100000) 4 :trim t))
  (assert-equal "0.00001" (rational-to-string (/ 1 100000) 5 :trim t))
  (assert-equal "0.00001" (rational-to-string (/ 1 100000) 6 :trim t))
  (assert-equal "0.00001" (rational-to-string (/ 1 100000) 6 :trim t))

  (assert-equal "0"       (rational-to-string (/ 1 10) 0 :trim t))
  (assert-equal "0.1"     (rational-to-string (/ 1 10) 1 :trim t))
  (assert-equal "0.1"     (rational-to-string (/ 1 10) 2 :trim t))

  (assert-equal "0"       (rational-to-string (/ 49999 100000) 0 :trim t))
  (assert-equal "0.5"     (rational-to-string (/ 49999 100000) 1 :trim t))
  (assert-equal "0.5"     (rational-to-string (/ 49999 100000) 2 :trim t))
  (assert-equal "0.5"     (rational-to-string (/ 49999 100000) 3 :trim t))
  (assert-equal "0.5"     (rational-to-string (/ 49999 100000) 4 :trim t))
  (assert-equal "0.49999" (rational-to-string (/ 49999 100000) 5 :trim t))

  (assert-equal "2"       (rational-to-string (/ 3 2) 0 :trim t))
  (assert-equal "1.5"     (rational-to-string (/ 3 2) 1 :trim t))
  (assert-equal "1.5"     (rational-to-string (/ 3 2) 2 :trim t))

  (assert-equal "100000"  (rational-to-string (/ 199999 2) 0 :trim t))
  (assert-equal "99999.5" (rational-to-string (/ 199999 2) 1 :trim t))
  (assert-equal "99999.5" (rational-to-string (/ 199999 2) 2 :trim t))

  (assert-equal "1.235"   (rational-to-string (/ 12345 10000) 3  :trim t))
  (assert-equal "-1.235"  (rational-to-string (/ -12345 10000) 3  :trim t))
  (assert-equal "1.2346"  (rational-to-string (/ 123455 100000) 4  :trim t))
  (assert-equal "-1.2346" (rational-to-string (/ -123455 100000) 4  :trim t)))

(define-test decimal-placement
  (assert-equal "0.00001" (rational-to-string (expt 10 -5) 10 :trim t))
  (assert-equal  "0.0001" (rational-to-string (expt 10 -4) 10 :trim t))
  (assert-equal   "0.001" (rational-to-string (expt 10 -3) 10 :trim t))
  (assert-equal    "0.01" (rational-to-string (expt 10 -2) 10 :trim t))
  (assert-equal     "0.1" (rational-to-string (expt 10 -1) 10 :trim t))
  (assert-equal       "1" (rational-to-string (expt 10  0) 10 :trim t))
  (assert-equal      "10" (rational-to-string (expt 10  1) 10 :trim t))
  (assert-equal     "100" (rational-to-string (expt 10  2) 10 :trim t))
  (assert-equal    "1000" (rational-to-string (expt 10  3) 10 :trim t))
  (assert-equal   "10000" (rational-to-string (expt 10  4) 10 :trim t))
  (assert-equal  "100000" (rational-to-string (expt 10  5) 10 :trim t)))

(define-test harmonic-series-test
  "Test (harmonic-series 250)."
  (assert-equal
   "6.1006752494325792775723270159741508963779612275122648211523647985"
   (rational-to-string (harmonic-series 250) 64)))

(define-test rational-to-string-test
  (assert-equal "0.02" (rational-to-string (/ 15 1000) 2 :sci nil :trim nil))
  (assert-equal "0.002" (rational-to-string (/ 15 10000) 3 :sci nil :trim nil)))

;;;
;;; (string-to-rational)
;;;

(define-test invalid-characters
  (assert-equal (values nil "foo")
                (string-to-rational "foo"))
  (assert-equal (values 123 "x")
                (string-to-rational "123x"))
  (assert-equal (values nil "x123")
                (string-to-rational "x123"))
  (assert-equal (values 1 "x3")
                (string-to-rational "1x3"))
  (assert-equal (values nil "x.123")
                (string-to-rational "x.123"))
  (assert-equal (values 0 "x.123")
                (string-to-rational "0x.123"))
  (assert-equal (values 0 "x123")
                (string-to-rational "0.x123"))
  (assert-equal (values (/ 123 1000) "x")
                (string-to-rational "0.123x")))

(define-test leading-implicit-positive-sign-test
  (assert-equal (values nil ".")
                (string-to-rational "."))
  (assert-equal (values nil "..1")
                (string-to-rational "..1"))
  (assert-equal (values 0 ".1")
                (string-to-rational "0..1"))
  (assert-equal (values (/ 10) "")
                 (string-to-rational ".1"))
  (assert-equal (values 1 "")
                (string-to-rational "1"))
  (assert-equal (values 0 "")
                (string-to-rational "0"))
  (assert-equal (values 0 "")
                (string-to-rational "00"))
  (assert-equal (values 0 "")
                (string-to-rational "000"))
  (assert-equal (values 0 "")
                (string-to-rational "0000"))
  (assert-equal (values (/ 10) "")
                 (string-to-rational ".1"))
  (assert-equal (values (/ 10) "")
                 (string-to-rational "0.1"))
  (assert-equal (values (/ 10) "")
                 (string-to-rational "00.1"))
  (assert-equal (values (/ 12 100) "")
                 (string-to-rational ".12"))
  (assert-equal (values (/ 12 100) "")
                 (string-to-rational "0.12"))
  (assert-equal (values (/ 12 100) "")
                 (string-to-rational "00.12"))
  (assert-equal (values 1 "")
                (string-to-rational "1"))
  (assert-equal (values 1 "")
                (string-to-rational "01"))
  (assert-equal (values 1 "")
                (string-to-rational " 001"))
  (assert-equal (values 1 "")
                (string-to-rational "001"))
  (assert-equal (values 1 "")
                (string-to-rational "0001"))
  (assert-equal (values 1 "")
                (string-to-rational "00001")))

(define-test leading-positive-sign-test
  (assert-equal (values nil "+.")
                (string-to-rational "+."))
  (assert-equal (values nil ".+")
                (string-to-rational ".+"))
  (assert-equal (values 1 "+")
                (string-to-rational "1+"))
  (assert-equal (values nil ".+1")
                (string-to-rational ".+1"))
  (assert-equal (values (/ 10) "+")
                (string-to-rational ".1+"))
  (assert-equal (values (/ 10) " +")
                (string-to-rational ".1 +"))
  (assert-equal (values nil "+..1")
                (string-to-rational "+..1"))
  (assert-equal (values 0 ".1")
                (string-to-rational "+0..1"))
  (assert-equal (values nil "++1")
                (string-to-rational "++1"))
  (assert-equal (values 0 "")
                (string-to-rational "+0"))
  (assert-equal (values 0 "")
                (string-to-rational "+00"))
  (assert-equal (values 0 "")
                (string-to-rational "+000"))
  (assert-equal (values 0 "")
                (string-to-rational "+0000"))
  (assert-equal (values (/ 10) "")
                 (string-to-rational "+.1"))
  (assert-equal (values (/ 10) "")
                 (string-to-rational "+0.1"))
  (assert-equal (values (/ 10) "")
                 (string-to-rational "+00.1"))
  (assert-equal (values (/ 12 100) "")
                 (string-to-rational "+.12"))
  (assert-equal (values (/ 12 100) "")
                 (string-to-rational "+0.12"))
  (assert-equal (values (/ 12 100) "")
                 (string-to-rational "+00.12"))
  (assert-equal (values 1 "")
                (string-to-rational "+1"))
  (assert-equal (values 1 "")
                (string-to-rational "+01"))
  (assert-equal (values 1 "")
                (string-to-rational " +001"))
  (assert-equal (values 1 "")
                (string-to-rational "+001"))
  (assert-equal (values 1 "")
                (string-to-rational "+0001"))
  (assert-equal (values 1 "")
                (string-to-rational "+00001")))

(define-test leading-negative-sign-test
  (assert-equal (values nil "-.")
                (string-to-rational "-."))
  (assert-equal (values nil ".-")
                (string-to-rational ".-"))
  (assert-equal (values 1 "-")
                (string-to-rational "1-"))
  (assert-equal (values nil ".-1")
                (string-to-rational ".-1"))
  (assert-equal (values (/ 10) "-")
                (string-to-rational ".1-"))
  (assert-equal (values (/ 10) " -")
                (string-to-rational ".1 -"))
  (assert-equal (values nil ".-1")
                (string-to-rational ".-1"))
  (assert-equal (values nil "-..1")
                (string-to-rational "-..1"))
  (assert-equal (values 0 ".1")
                (string-to-rational "-0..1"))
  (assert-equal (values nil "--1")
                (string-to-rational "--1"))
  (assert-equal (values 0 "")
                (string-to-rational "-0"))
  (assert-equal (values 0 "")
                (string-to-rational "-00"))
  (assert-equal (values 0 "")
                (string-to-rational "-000"))
  (assert-equal (values 0 "")
                (string-to-rational "-0000"))
  (assert-equal (values (/ -10) "")
                (string-to-rational "-.1"))
  (assert-equal (values (/ -10) "")
                (string-to-rational "-0.1"))
  (assert-equal (values (/ -10) "")
                (string-to-rational "-00.1"))
  (assert-equal (values (/ -12 100) "")
                (string-to-rational "-.12"))
  (assert-equal (values (/ -12 100) "")
                (string-to-rational "-0.12"))
  (assert-equal (values (/ -12 100) "")
                (string-to-rational "-00.12"))
  (assert-equal (values -1 "")
                (string-to-rational "-1"))
  (assert-equal (values -1 "")
                (string-to-rational "-01"))
  (assert-equal (values -1 "")
                (string-to-rational " -001"))
  (assert-equal (values -1 "")
                (string-to-rational "-001"))
  (assert-equal (values -1 "")
                (string-to-rational "-0001"))
  (assert-equal (values -1 "")
                (string-to-rational "-00001")))

(define-test space-test
  ;; empty strings
  (assert-equal (values nil "")
                (string-to-rational ""))
  (assert-equal (values nil " ")
                (string-to-rational " "))
  (assert-equal (values nil "  ")
                (string-to-rational "  "))
  (assert-equal (values nil "   ")
                (string-to-rational "   "))

  ;; leading space only
  (assert-equal (values 1 "")
                (string-to-rational "   1"))
  (assert-equal (values 1 "")
                (string-to-rational "   1."))
  (assert-equal (values 12 "")
                (string-to-rational "   12"))
  (assert-equal (values 12 "")
                (string-to-rational "   12."))
  (assert-equal (values 123 "")
                (string-to-rational "   123"))
  (assert-equal (values 123 "")
                (string-to-rational "   123."))
  (assert-equal (values (+ 123 (/ 4 10)) "")
                (string-to-rational "   123.4"))
  (assert-equal (values (+ 123 (/ 45 100)) "")
                   (string-to-rational "   123.45"))

  (assert-equal (values 1 "+")
                (string-to-rational "   1+"))
  (assert-equal (values 1 "+")
                (string-to-rational "   1.+"))
  (assert-equal (values 1 " +")
                (string-to-rational "   1. +"))
  (assert-equal (values 1 "")
                (string-to-rational "   +1"))
  (assert-equal (values 1 "")
                (string-to-rational "   +1."))
  (assert-equal (values 12 "")
                (string-to-rational "   +12"))
  (assert-equal (values 12 "")
                (string-to-rational "   +12."))
  (assert-equal (values 123 "")
                (string-to-rational "   +123"))
  (assert-equal (values 123 "")
                (string-to-rational "   +123."))
  (assert-equal (values (+ 123 (/ 4 10)) "")
                   (string-to-rational "   +123.4"))
  (assert-equal (values (+ 123 (/ 45 100)) "")
                   (string-to-rational "   +123.45"))

  (assert-equal (values 1 "-")
                (string-to-rational "   1-"))
  (assert-equal (values 1 "-")
                (string-to-rational "   1.-"))
  (assert-equal (values 1 " -")
                (string-to-rational "   1. -"))
  (assert-equal (values -1 "")
                (string-to-rational "   -1"))
  (assert-equal (values -1 "")
                (string-to-rational "   -1."))
  (assert-equal (values -12 "")
                (string-to-rational "   -12"))
  (assert-equal (values -12 "")
                (string-to-rational "   -12."))
  (assert-equal (values -123 "")
                (string-to-rational "   -123"))
  (assert-equal (values -123 "")
                (string-to-rational "   -123."))
  (assert-equal (values (- -123 (/ 4 10)) "")
                  (string-to-rational "   -123.4"))
  (assert-equal (values (- -123 (/ 45 100)) "")
                  (string-to-rational "   -123.45"))

  ; both leading and trailing space
  (assert-equal (values 1 "   ")
                (string-to-rational "   1   "))
  (assert-equal (values 1 "   ")
                (string-to-rational "   1.   "))
  (assert-equal (values 12 "   ")
                (string-to-rational "   12   "))
  (assert-equal (values 12 "   ")
                (string-to-rational "   12.   "))
  (assert-equal (values 123 "   ")
                (string-to-rational "   123   "))
  (assert-equal (values 123 "   ")
                (string-to-rational "   123.   "))
  (assert-equal (values (+ 123 (/ 4 10)) "   ")
                   (string-to-rational "   123.4   "))
  (assert-equal (values (+ 123 (/ 45 100)) "   ")
                   (string-to-rational "   123.45   "))

  (assert-equal (values 1 "+   ")
                (string-to-rational "   1+   "))
  (assert-equal (values 1 "+   ")
                (string-to-rational "   1.+   "))
  (assert-equal (values 1 " +   ")
                (string-to-rational "   1. +   "))
  (assert-equal (values 1 "   ")
                (string-to-rational "   +1   "))
  (assert-equal (values 1 "   ")
                (string-to-rational "   +1.   "))
  (assert-equal (values 12 "   ")
                (string-to-rational "   +12   "))
  (assert-equal (values 12 "   ")
                (string-to-rational "   +12.   "))
  (assert-equal (values 123 "   ")
                (string-to-rational "   +123   "))
  (assert-equal (values 123 "   ")
                (string-to-rational "   +123.   "))
  (assert-equal (values (+ 123 (/ 4 10)) "   ")
                   (string-to-rational "   +123.4   "))
  (assert-equal (values (+ 123 (/ 45 100)) "   ")
                   (string-to-rational "   +123.45   "))

  (assert-equal (values 1 "-   ")
                (string-to-rational "   1-   "))
  (assert-equal (values 1 "-   ")
                (string-to-rational "   1.-   "))
  (assert-equal (values 1 " -   ")
                (string-to-rational "   1. -   "))
  (assert-equal (values -1 " ")
                (string-to-rational "   -1 "))
  (assert-equal (values -1 "   ")
                (string-to-rational "   -1.   "))
  (assert-equal (values -12 "   ")
                (string-to-rational "   -12   "))
  (assert-equal (values -12 "   ")
                (string-to-rational "   -12.   "))
  (assert-equal (values -123 "   ")
                (string-to-rational "   -123   "))
  (assert-equal (values -123 "   ")
                (string-to-rational "   -123.   "))
  (assert-equal (values (- -123 (/ 4 10)) "   ")
                  (string-to-rational "   -123.4   "))
  (assert-equal (values (- -123 (/ 45 100)) "   ")
                  (string-to-rational "   -123.45   "))

  ;; trailing space only
  (assert-equal (values 1 "   ")
                (string-to-rational "1   "))
  (assert-equal (values 1 "   ")
                (string-to-rational "1.   "))
  (assert-equal (values 12 "   ")
                (string-to-rational "12   "))
  (assert-equal (values 12 "   ")
                (string-to-rational "12.   "))
  (assert-equal (values 123 "   ")
                (string-to-rational "123   "))
  (assert-equal (values 123 "   ")
                (string-to-rational "123.   "))
  (assert-equal (values (+ 123 (/ 4 10)) "   ")
                   (string-to-rational "123.4   "))
  (assert-equal (values (+ 123 (/ 45 100)) "   ")
                   (string-to-rational "123.45   "))

  (assert-equal (values 1 "+   ")
                (string-to-rational "1+   "))
  (assert-equal (values 1 "+   ")
                (string-to-rational "1.+   "))
  (assert-equal (values 1 " +   ")
                (string-to-rational "1. +   "))
  (assert-equal (values 1 "   ")
                (string-to-rational "+1   "))
  (assert-equal (values 1 "   ")
                (string-to-rational "+1.   "))
  (assert-equal (values 12 "   ")
                (string-to-rational "+12   "))
  (assert-equal (values 12 "   ")
                (string-to-rational "+12.   "))
  (assert-equal (values 123 "   ")
                (string-to-rational "+123   "))
  (assert-equal (values 123 "   ")
                (string-to-rational "+123.   "))
  (assert-equal (values (+ 123 (/ 4 10)) "   ")
                   (string-to-rational "+123.4   "))
  (assert-equal (values (+ 123 (/ 45 100)) "   ")
                   (string-to-rational "+123.45   "))

  (assert-equal (values 1 "-   ")
                (string-to-rational "1-   "))
  (assert-equal (values 1 "-   ")
                (string-to-rational "1.-   "))
  (assert-equal (values 1 " -   ")
                (string-to-rational "1. -   "))
  (assert-equal (values -1 "   ")
                (string-to-rational "-1   "))
  (assert-equal (values -1 "   ")
                (string-to-rational "-1.   "))
  (assert-equal (values -12 "   ")
                (string-to-rational "-12   "))
  (assert-equal (values -12 "   ")
                (string-to-rational "-12.   "))
  (assert-equal (values -123 "   ")
                (string-to-rational "-123   "))
  (assert-equal (values -123 "   ")
                (string-to-rational "-123.   "))
  (assert-equal (values (- -123 (/ 4 10)) "   ")
                 (string-to-rational "-123.4   "))
  (assert-equal (values (- -123 (/ 45 100)) "   ")
                 (string-to-rational "-123.45   ")))

(define-test exponential-input-test
  ;; A trailing "e" or "E" without an exponent is just junk.
  (assert-equal (values (+ 1 (/ 23 100)) "e")
                (string-to-rational "1.23e"))
  (assert-equal (values (+ 1 (/ 23 100)) "E")
                (string-to-rational "1.23E"))
  (assert-equal (values (+ 1 (/ 23 100)) "e+")
                (string-to-rational "1.23e+"))
  (assert-equal (values (+ 1 (/ 23 100)) "E+")
                (string-to-rational "1.23E+"))
  (assert-equal (values (+ 1 (/ 23 100)) "e-")
                (string-to-rational "1.23e-"))
  (assert-equal (values (+ 1 (/ 23 100)) "E-")
                (string-to-rational "1.23E-"))
  (assert-equal (values 123 "")
                (string-to-rational "1.23e2"))
  (assert-equal (values 123 "")
                (string-to-rational "1.23E2"))
  (assert-equal (values (+ 123 (/ 4 10)) "")
                (string-to-rational "1.234e2"))
  (assert-equal (values (+ 123 (/ 4 10)) "")
                (string-to-rational "1.234E2"))
  (assert-equal (values (+ 123 (/ 4 10)) "")
                (string-to-rational "1.234e+2"))
  (assert-equal (values (+ 123 (/ 4 10)) "")
                (string-to-rational "1.234E+2"))
  (assert-equal (values (/ 1234 100000) "")
                (string-to-rational "1.234e-2"))
  (assert-equal (values (/ 1234 100000) "")
                (string-to-rational "1.234E-2"))
  (assert-equal (values 123 "")
                (string-to-rational "1.23e02"))
  (assert-equal (values 123 "")
                (string-to-rational "1.23E02"))
  (assert-equal (values 123 "")
                (string-to-rational "1.23e+02"))
  (assert-equal (values 123 "")
                (string-to-rational "1.23E+02"))
  (assert-equal (values 123 "")
                (string-to-rational "1.23e002"))
  (assert-equal (values 123 "")
                (string-to-rational "1.23E002"))
  (assert-equal (values 123 "")
                (string-to-rational "1.23e+002"))
  (assert-equal (values 123 "")
                (string-to-rational "1.23E+002"))
  (assert-equal (values 123 "")
                (string-to-rational "+1.23e2"))
  (assert-equal (values 123 "")
                (string-to-rational "+1.23E2"))
  (assert-equal (values 123 "")
                (string-to-rational "+1.23e+2"))
  (assert-equal (values 123 "")
                (string-to-rational "+1.23E+2"))
  (assert-equal (values -123 "")
                (string-to-rational "-1.23e2"))
  (assert-equal (values -123 "")
                (string-to-rational "-1.23E2"))
  (assert-equal (values (/ -1234 100000) "")
                (string-to-rational "-1.234e-2"))
  (assert-equal (values (/ -1234 100000) "")
                (string-to-rational "-1.234E-2"))
  (assert-equal (values (* 123 (expt 10 456)) "")
                (string-to-rational "+123e456"))
  (assert-equal (values (* 123 (expt 10 456)) "")
                (string-to-rational "+123E456")))

(defun main ()
  (let ((*print-failures* t)
        (*print-errors* t))
    (run-tests :all :rational-to-string-test)
    (fresh-line)))
